import { useState, useRef, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'
import { Tent, Bed, UtensilsCrossed, Compass, Calendar, CheckCircle, CheckCircle2, Sparkles, Star, ArrowRight, Shield, Clock, Award, Users } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Rent.css'

const Rent = () => {
  const navigate = useNavigate()
  const [selectedCategory, setSelectedCategory] = useState('')
  const [returnDate, setReturnDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const categories = [
    { 
      id: 'tents', 
      name: 'Палатки', 
      icon: Tent,
      color: '#0F766E',
      description: 'Палатки различных размеров для комфортного отдыха на природе. От 2 до 8 мест.',
      items: ['Палатка 2-местная', 'Палатка 4-местная', 'Палатка 6-местная', 'Палатка 8-местная']
    },
    { 
      id: 'sleeping', 
      name: 'Спальные системы', 
      icon: Bed,
      color: '#D97706',
      description: 'Спальники, коврики и подушки для комфортного сна в походе.',
      items: ['Спальник -10°C', 'Спальник 0°C', 'Спальник +10°C', 'Коврик самонадувающийся']
    },
    { 
      id: 'kitchen', 
      name: 'Кухня', 
      icon: UtensilsCrossed,
      color: '#84CC16',
      description: 'Посуда, горелки, котелки и другое оборудование для приготовления пищи.',
      items: ['Газовая горелка', 'Котелок 2л', 'Набор посуды', 'Холодильник переносной']
    },
    { 
      id: 'navigation', 
      name: 'Навигация', 
      icon: Compass,
      color: '#78716C',
      description: 'Компасы, GPS-навигаторы и карты для ориентирования на местности.',
      items: ['GPS-навигатор', 'Компас', 'Карты местности', 'Рация']
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedCategory || !returnDate) {
      alert('Пожалуйста, выберите категорию и дату возврата')
      return
    }

    const saved = localStorage.getItem('campingRentals')
    const rentals = saved ? JSON.parse(saved) : []
    
    const categoryData = categories.find(c => c.id === selectedCategory)
    const randomItem = categoryData.items[Math.floor(Math.random() * categoryData.items.length)]
    
    const today = new Date()
    const startDate = today.toISOString().split('T')[0]
    
    const newRental = {
      id: Date.now(),
      item: randomItem,
      category: selectedCategory,
      startDate: startDate,
      returnDate: returnDate,
      status: 'active',
      review: null
    }
    
    rentals.push(newRental)
    localStorage.setItem('campingRentals', JSON.stringify(rentals))
    
    setIsSubmitted(true)
    
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  return (
    <div className="rent-page-outdoor">
      <div className="container">
        <section className="rent-header-outdoor">
          <h1 className="rent-title-outdoor">Арендовать оборудование</h1>
          <p className="rent-description-outdoor">
            Выберите категорию оборудования и укажите дату возврата. Все оборудование проверено и готово к использованию.
          </p>
        </section>

        <section className="rent-form-section-outdoor">
          <form className="form-outdoor" onSubmit={handleSubmit}>
            <div className="categories-selection-outdoor">
              {categories.map((category) => {
                const Icon = category.icon
                return (
                  <button
                    key={category.id}
                    type="button"
                    className={`category-card-outdoor ${selectedCategory === category.id ? 'selected' : ''}`}
                    onClick={() => setSelectedCategory(category.id)}
                    style={{ '--category-color': category.color }}
                  >
                    <div className="category-icon-wrapper-outdoor">
                      <Icon size={36} />
                    </div>
                    <h3 className="category-name-outdoor">{category.name}</h3>
                    <p className="category-desc-outdoor">{category.description}</p>
                    {selectedCategory === category.id && (
                      <div className="check-badge-outdoor">
                        <CheckCircle size={20} />
                      </div>
                    )}
                  </button>
                )
              })}
            </div>

            {selectedCategory && (
              <div className="date-section-outdoor">
                <div className="date-label-outdoor">
                  <Calendar size={18} />
                  <span>Дата возврата оборудования</span>
                </div>
                <input
                  type="date"
                  className="date-input-outdoor"
                  value={returnDate}
                  onChange={(e) => setReturnDate(e.target.value)}
                  min={minDateStr}
                  max={maxDateStr}
                  required
                />
              </div>
            )}

            <button 
              type="submit" 
              className="submit-btn-outdoor" 
              disabled={!selectedCategory || !returnDate || isSubmitted}
            >
              <span>Оформить аренду</span>
            </button>

            {isSubmitted && (
              <div className="success-message-outdoor">
                <CheckCircle size={28} />
                <span>Аренда успешно оформлена!</span>
              </div>
            )}
          </form>
        </section>

        <ImageSlider />

        <EquipmentFeaturesSection />
        
        <PopularEquipmentSection />
        
        <WhyChooseUsSection />
      </div>
    </div>
  )
}

const EquipmentFeaturesSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const features = [
    {
      icon: CheckCircle2,
      title: 'Проверено экспертами',
      description: 'Все оборудование проходит тщательную проверку перед выдачей',
      color: 'var(--outdoor-teal)'
    },
    {
      icon: Sparkles,
      title: 'Премиум качество',
      description: 'Только проверенные бренды и модели от ведущих производителей',
      color: 'var(--outdoor-amber)'
    },
    {
      icon: Star,
      title: 'Гарантия сервиса',
      description: 'Круглосуточная поддержка и помощь в любой ситуации',
      color: 'var(--outdoor-sage)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`features-section-outdoor ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="features-header-outdoor">
          <h2 className="features-title-outdoor">Почему выбирают нас</h2>
          <p className="features-description-outdoor">
            Мы гарантируем качество и надежность каждого элемента оборудования
          </p>
        </div>
        <div className="features-grid-outdoor">
          {features.map((feature, index) => {
            const Icon = feature.icon
            return (
              <div 
                key={index} 
                className="feature-card-outdoor"
                style={{ 
                  animationDelay: `${index * 0.15}s`,
                  '--feature-color': feature.color
                }}
              >
                <div className="feature-icon-wrapper-outdoor">
                  <Icon size={32} />
                </div>
                <h3 className="feature-title-outdoor">{feature.title}</h3>
                <p className="feature-text-outdoor">{feature.description}</p>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

const PopularEquipmentSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const equipment = [
    {
      name: 'Палатка MSR Hubba Hubba',
      category: 'Палатки',
      rating: 4.9,
      rentals: 234,
      price: 'от 1500₽/день',
      features: ['2-местная', 'Водонепроницаемая', 'Легкая']
    },
    {
      name: 'Спальник Marmot Trestles',
      category: 'Спальные системы',
      rating: 4.8,
      rentals: 189,
      price: 'от 800₽/день',
      features: ['Комфорт -10°C', 'Синтетика', 'Компактный']
    },
    {
      name: 'Рюкзак Osprey Atmos',
      category: 'Рюкзаки',
      rating: 4.9,
      rentals: 156,
      price: 'от 1200₽/день',
      features: ['65 литров', 'Эргономичный', 'Прочный']
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`popular-equipment-outdoor ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="popular-equipment-header-outdoor">
          <h2 className="popular-equipment-title-outdoor">Популярное оборудование</h2>
          <p className="popular-equipment-description-outdoor">
            Проверенное временем и тысячами путешественников
          </p>
        </div>
        <div className="popular-equipment-grid-outdoor">
          {equipment.map((item, index) => (
            <div 
              key={index} 
              className="equipment-card-outdoor"
              style={{ animationDelay: `${index * 0.1}s` }}
            >
              <div className="equipment-badge-outdoor">{item.category}</div>
              <h3 className="equipment-name-outdoor">{item.name}</h3>
              <div className="equipment-rating-outdoor">
                <div className="equipment-stars-outdoor">
                  {Array.from({ length: 5 }).map((_, i) => (
                    <Star
                      key={i}
                      size={16}
                      fill={i < Math.floor(item.rating) ? '#D97706' : 'none'}
                      color={i < Math.floor(item.rating) ? '#D97706' : '#A8A29E'}
                    />
                  ))}
                </div>
                <span className="equipment-rating-value-outdoor">{item.rating}</span>
                <span className="equipment-rentals-outdoor">({item.rentals} аренд)</span>
              </div>
              <div className="equipment-features-outdoor">
                {item.features.map((feature, i) => (
                  <span key={i} className="equipment-feature-outdoor">{feature}</span>
                ))}
              </div>
              <div className="equipment-price-outdoor">{item.price}</div>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

const WhyChooseUsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const benefits = [
    {
      icon: Shield,
      title: 'Гарантия качества',
      description: 'Все оборудование проходит тщательную проверку перед каждой арендой',
      color: 'var(--outdoor-teal)'
    },
    {
      icon: Clock,
      title: 'Быстрая доставка',
      description: 'Доставим оборудование в удобное для вас время и место',
      color: 'var(--outdoor-amber)'
    },
    {
      icon: Award,
      title: 'Премиум бренды',
      description: 'Только проверенные производители с мировым именем',
      color: 'var(--outdoor-sage)'
    },
    {
      icon: Users,
      title: 'Поддержка 24/7',
      description: 'Наша команда всегда готова помочь в любой ситуации',
      color: 'var(--outdoor-teal)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`why-choose-us-outdoor ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="why-choose-us-header-outdoor">
          <h2 className="why-choose-us-title-outdoor">Почему выбирают нас</h2>
          <p className="why-choose-us-description-outdoor">
            Мы делаем ваши приключения безопасными и комфортными
          </p>
        </div>
        <div className="why-choose-us-grid-outdoor">
          {benefits.map((benefit, index) => {
            const Icon = benefit.icon
            return (
              <div 
                key={index} 
                className="benefit-card-outdoor"
                style={{ 
                  animationDelay: `${index * 0.1}s`,
                  '--benefit-color': benefit.color
                }}
              >
                <div className="benefit-icon-wrapper-outdoor">
                  <Icon size={32} />
                </div>
                <h3 className="benefit-title-outdoor">{benefit.title}</h3>
                <p className="benefit-text-outdoor">{benefit.description}</p>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

export default Rent

